import argparse
import grpc
import json
import logging

from nuance.nlu.wordset.v1beta1.wordset_pb2 import *
from nuance.nlu.wordset.v1beta1.wordset_pb2_grpc import *
from nuance.nlu.common.v1beta1.resource_pb2 import *

from google.protobuf.json_format import MessageToJson

log = logging.getLogger(__name__)

def parse_args():
    parser = argparse.ArgumentParser(
        prog="client.py",
        usage="%(prog)s <command> [-options]",
        add_help=False,
        formatter_class=lambda prog: argparse.HelpFormatter(
            prog, max_help_position=45, width=100)
    )

    parser.add_argument('command', metavar='command', nargs='?',
        choices=['compile', 'get-metadata', 'delete'], default='compile',
        help="Command to execute [values: compile, get-metadata, delete] (default: compile)")
    options = parser.add_argument_group("options")
    options.add_argument("-h", "--help", action="help",
                         help="Show this help message and exit.")
    options.add_argument("--token", nargs="?", help=argparse.SUPPRESS)
    options.add_argument("-s", "--serverUrl", metavar="url", nargs="?",
                         help="NLU server URL, default=localhost:9090.", default='localhost:9090')
    options.add_argument("--wordsetFile", type=argparse.FileType("r"), metavar="file",
                         nargs="?", help="Wordset JSON file.")
    options.add_argument("--artifactUrn", nargs="?", metavar="urn", help="Compiled Wordset URN.")
    options.add_argument("--modelUrn", nargs="?", metavar="urn", help="NLU Model URN.")
    options.add_argument("--metadata", metavar="metadata", nargs="+", default=[],
                         help="Wordset metadata defined as one or more key:value pairs.")
    options.add_argument("--clientData", metavar="clientData", nargs="+", default=[],
                         help="Client data defined as one or more key=value pairs.")
    return parser.parse_args()

def create_channel(args):
    channel = None
    call_credentials = None

    if args.token:
        log.debug("Adding CallCredentials with token %s" % args.token)
        call_credentials = grpc.access_token_call_credentials(args.token)

    log.debug("Creating secure gRPC channel")
    channel_credentials = grpc.ssl_channel_credentials()
    channel_credentials = grpc.composite_channel_credentials(channel_credentials, call_credentials)
    channel = grpc.secure_channel(args.serverUrl, credentials=channel_credentials)

    return channel

def create_get_wordset_metadata_request(artifactUrn):
    artifact_reference = ResourceReference(uri=artifactUrn)
    return GetWordsetMetadataRequest(artifact_reference=artifact_reference)

def create_delete_wordset_request(artifactUrn):
    artifact_reference = ResourceReference(uri=artifactUrn)
    return DeleteWordsetRequest(artifact_reference=artifact_reference)

def list_to_dict(list, separator = ':'):
  return dict(entry.split(separator) for entry in list)

def create_compile_wordset_request(args):
  target_artifact_reference = ResourceReference(uri=args.artifactUrn)
  companion_artifact_reference = ResourceReference(uri=args.modelUrn)
  wordset = json.dumps(json.load(args.wordsetFile))
  return CompileWordsetRequest(
    target_artifact_reference=target_artifact_reference,
    companion_artifact_reference=companion_artifact_reference,
    wordset=wordset,
    metadata=list_to_dict(args.metadata),
    client_data=list_to_dict(args.clientData))

def compile_wordset(args):
  with create_channel(args) as channel:
    stub = WordsetStub(channel)
    compiled_wordset_request = create_compile_wordset_request(args)
    for message in stub.CompileWordsetAndWatch(compiled_wordset_request):
      print(MessageToJson(message))

def get_wordset_metadata(args):
  with create_channel(args) as channel:
    stub = WordsetStub(channel)
    response = stub.GetWordsetMetadata(create_get_wordset_metadata_request(args.artifactUrn))
    print(MessageToJson(response))

def delete_wordset(args):
  with create_channel(args) as channel:
    stub = WordsetStub(channel)
    response = stub.DeleteWordset(create_delete_wordset_request(args.artifactUrn))
    print(MessageToJson(response))

def main():
  args = parse_args()
  log_level = logging.DEBUG
  logging.basicConfig(
    format='%(lineno)d %(asctime)s %(levelname)-5s: %(message)s', level=log_level)
  switcher = {
    'compile' : compile_wordset,
    'get-metadata' : get_wordset_metadata,
    'delete' : delete_wordset
  }
  switcher.get(args.command)(args)
  print("Done")

if __name__ == '__main__':
  main()